/**
* \file: errmem_interface.c
*
* This file defines general funcitonality used by the error memory daemon.
* 
* This file implements functionality to start and stop the daemon and
* to destroy the backends
*
* \component: errmemd
*
* \author: Kai Tomerius (ktomerius@de.adit-jv.com)
*          Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/

#include <ctype.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <sys/types.h>
#include <sys/epoll.h>
#include <unistd.h>
#include <syslog.h>
#include <linux/errmem.h>

#include "errmem_debug.h"
#include "errmem_interface.h"
#include "errmemd.h"

// errmem_get_version - get a driver parameter
unsigned long errmem_get_version(ErrmemInterface_t* b)
{
	if (b) {
		unsigned long value;
		int rc = _ioctl(b->efd,
				IOCTL_ERRMEM_VERSION, &value);
		return rc>=0 ? value : (unsigned long) ~0;
	}
	return (unsigned long) 0;
}

// errmem_get_number_of_slots - get a driver parameter
unsigned errmem_get_number_of_slots(ErrmemInterface_t* b)
{
	if (b) {
		unsigned long value;
		int rc = _ioctl(b->efd,
				IOCTL_ERRMEM_GET_NUMBER_OF_SLOTS, &value);
		return rc>=0 ? (unsigned) value : (unsigned) ~0;
	}

	return 0;
}

// errmem_create - initialize the error memory interface
ErrmemInterface_t* errmem_create(char* driver_dev)
{
	int32_t rc = 0;
	ErrmemInterface_t* b =
		(ErrmemInterface_t*) calloc(1, sizeof(ErrmemInterface_t));
	const char* name = "/dev/errmem";

	if (b) {
		if (driver_dev) {
			name = driver_dev;
		}
		b->efd = _open(name, O_NONBLOCK);
		if (b->efd>0) {
			// check the version of error memory Linux kernel driver
			unsigned long version = errmem_get_version(b);
			if (version<ERRMEM_REQUIRED_VERSION_MIN ||
				version>ERRMEM_REQUIRED_VERSION_MAX) {
				// the driver version is insufficient
				fprintf(stderr,
						"%s: wrong version v%lu.%02lx, "
						"expecting v%d.%02x ... v%d.%02x\n",
						name,
						version>>8, version&0xff,
						ERRMEM_REQUIRED_VERSION_MIN>>8,
						ERRMEM_REQUIRED_VERSION_MIN&0xff,
						ERRMEM_REQUIRED_VERSION_MAX>>8,
						ERRMEM_REQUIRED_VERSION_MAX&0xff);
			} else {
				b->name = strdup(name);
				/* be careful with sizeof (struct) - you may get the padding bytes included 
				 * if the struct is not declared packed 
				 */
				b->maxsize = offsetof(struct errmem_message, message) + ERRMEM_MAX_ENTRY_LENGTH;
				b->low = 0;
				b->flush = 0;
				b->buffer = calloc(1, b->maxsize);
				b->con = NULL;
				rc = _ioctl(b->efd, IOCTL_ERRMEM_SET_READ_SIZE, &b->maxsize);
				if (rc < 0)
					syslog(LOG_CRIT, "%s %s", "IOCTL failed to set read size ", name);
				b->nr_drv_slots = errmem_get_number_of_slots(b);
				if (!b->nr_drv_slots || !~b->nr_drv_slots)
					syslog(LOG_CRIT, "%s %s", "IOCTL failed to get number of driver slots ", name);
			}
		} else {
			// failed to open the device
			syslog(LOG_CRIT, "%s %s", "failed to open frontend ", name);
		}
	}
	if (!b || !b->name || !b->buffer || rc < 0) {
		syslog(LOG_CRIT, "%s %s", "Interface to frontend could not be created ", name);
		errmem_destroy(b);
		b = NULL;
	}

	return b;
}

// errmem_info - information about the error memory device
void errmem_info(ErrmemInterface_t* b, FILE* fd)
{
	if (b) {
		unsigned long version = errmem_get_version(b);
		int32_t fildes   = fileno(fd);

		dprintf(fildes, "\n%s", b->name);
		if (~version)
			dprintf(fildes, " v%lu.%02lx", version>>8, version&0xff);
		dprintf(fildes, ": %u slots",
			errmem_get_number_of_slots(b));
		dprintf(fildes, " of %u bytes,",
			errmem_get_slotsize(b));
		dprintf(fildes, " %u used,",
				errmem_get_used_slots(b));
		dprintf(fildes, " watermark low %u,",
			errmem_get_watermark_low(b));
		dprintf(fildes, " watermark high %u,",
			errmem_get_watermark_high(b));
		dprintf(fildes, " kernel message level %u\n",
			errmem_get_kernel_message_level(b));
	}
}

// mkmsg - cast to struct errmem_message* (and avoid a lint warning)
static inline struct errmem_message* mkmsg(unsigned char* m, unsigned o)
{
	return (struct errmem_message*) addptr(m, o);
}

int32_t errmem_connect_frontend(Errmemd_t *d)
{
	int32_t err = 0;
	struct epoll_event ev = {.events = EPOLLIN};
	if (d && d->ifc && d->is) {
		d->ifc->con = malloc(sizeof(ErrmemConnect_t));
		if (d->ifc->con) {
			memset(d->ifc->con, 0xFF, sizeof(ErrmemConnect_t));
			((ErrmemConnect_t*)d->ifc->con)->h = errmem_handle_frontend;
			((ErrmemConnect_t*)d->ifc->con)->sid = d->ifc->efd;
			ev.data.ptr = d->ifc->con;
			err = epoll_ctl(d->is->e, EPOLL_CTL_ADD, d->ifc->efd, &ev);
			if (-1 == err) {
				err = -errno;
				syslog(LOG_CRIT, "%s: %s", "Cannot connect to frontend "
					   "- add file handle to poll set failed - error = ", strerror(-err));
			}
		} else {
			err = -ENOMEM;
			syslog(LOG_CRIT, "%s: %s", "Cannot connect to frontend "
				   "- failed to allocate memory for context struture - error = ", strerror(-err));
		}
	} else {
		err = -EIO;
		syslog(LOG_CRIT, "%s: %s", "Cannot connect to frontend "
			   "- parameter check failed - error = ", strerror(-err));
	}

	return err;
}

int32_t errmem_disconnect_frontend(Errmemd_t *d)
{
	int32_t err = 0;
	if (d && d->ifc) {
		if (d->ifc->efd >= 0) {
			err = epoll_ctl(d->is->e, EPOLL_CTL_DEL, d->ifc->efd, NULL);
			if (-1 == err)
				err = -errno;
		} else
			err = -EBADF;
		if (d->ifc->con) {
			free(d->ifc->con);
			d->ifc->con = NULL;
		}
	} else 
		err = -EIO;

	if (err < 0)
		syslog(LOG_CRIT, "%s %s", "Error disconnecting from frontend. - error = ", strerror(-err));

	return err;
}


// errmem_flush - flush data from /dev/errmem to any error memory backend
static int32_t errmem_flush(Errmemd_t *d)
{
	int32_t err = 0;
	struct errmem_message* msg = NULL;
	/* Doing persistent mode read messages are always acknowledged
	 * In case mode switch to non persistent mode the value of d->out_ack
	 * determines this behaviour.
	 */
	int32_t ack = 1;

	if (d && d->ifc && d->ifc->buffer) {
		/* If the daemon is stopped or if no persistent backends are available
		 * no persistent access will be done throughout this lifetime of
		 * the daemon.
		 */
		if (d->stop || !d->backends)
			d->p_mode = 0;
		/* The daemon is stopped and no pure output devices specified.
		 * This stopps current execution and daemon disconnects from frontend.
		 * No further messages will be read from frontend until pure output
		 * devices are requested through the socket interface.
		 */
		if (d->stop && !d->outputs) {
			err = -ENOEXEC;
			syslog(LOG_CRIT, "%s %s", "Storing messages to persistent storages "
				   "stopped: error = ", strerror(-err));
			errmem_disconnect_frontend(d);
		}
		/* No backends, either persistent or pure output, specified. */
		if (!d->outputs && !d->backends) {
			err = -ENOMEDIUM;
			syslog(LOG_CRIT, "%s %s", "No storages specified at all "
				   ": error = ", strerror(-err));
			errmem_disconnect_frontend(d);
		}
		if (!err) {
			memset(d->ifc->buffer, 0xFF, d->ifc->maxsize);
			/* Read a message from the driver. */
			err = _ioctl(d->ifc->efd, IOCTL_ERRMEM_READ, d->ifc->buffer);

			if (0 < err) {
				msg = mkmsg(d->ifc->buffer, 0);

				if (msg->internal.seqnum == d->last_seq) {
					/* In case the daemon dumps the frontend
					 * to some device of class B in mode once
					 * without acknowledging the read messages
					 * we have to abort now.
					 * In this mode the daemon only reads messages
					 * calling the read function in a loop without
					 * using the poll function of the driver.
					 * Therefore it won't get the information that
					 * no more messages are available which is created
					 * inside the  poll function.
					 * Rather than it will get always the last
					 * readable message from the driver, even if this
					 * message is already flagged as read and delivered.
					 * The poll function sets the POLLIN if llrb_can_read
					 * determines a valid slot containing a readable next
					 * message. If not it doesn't set the POLLIN.
					 * The read function will only check for a next
					 * readable slot and updates the file handle's
					 * private data structure. It also uses the llrb_can_read
					 * funciton. In case llrb_can_read determines a next valid
					 * slot the private data structure is updated. In case
					 * llrb_read_next delivers no next valid slot, the private
					 * data structure remains unchanged. Therefore the
					 * value of the last message read keeps the same value and is
					 * requested again the next time the read function is called.
					 * The read function has no mechanism to signal the daemon
					 * that no more message are available.
					 * The frontend gets a next request for the same message
					 * and sees the concerned slot as not freed and deliver it,
					 * even if the read flag is already set. And that is ok. */
					if (!d->p_mode && d->out_once && !d->out_ack)
						err = 0;
				}
			}
			/* In case we got a message, store it to backends.
			 * err >  0 ==> valid message read
			 * err <  0 ==> Some error code
			 * err == 0 ==> no more message(s) available
			 */
			if (0 < err && msg) {
				/* Check sequence number consistency. Get last sequence number
				 * which is stored in the default device if possible.
				 * Block storages support this feature. They will deliver 
				 * either zero if no messages are currently stored in the
				 * persistent storage (e.g. after the storage was erased) or
				 * they will deliver the sequence number of that message which
				 * was stored successfully as the last message before the daemon
				 * stopped working the last time.
				 * All other backends will deliver zero. Class B devices can
				 * never deliver any valid sequence number because they don't
				 * store any messages peristently. File devices do not support
				 * this feature yet. The only disadvantage here is that the
				 * consistency between two life cycles of the error memory
				 * deamon (last message of life cycle one and first message of
				 * life cycle two) can not be checked in contrast to block
				 * storages.
				 */
				/* After a restart the sequence number has always to be 1 
				 * Whatever is aready stored in the persistent memory is
				 * not of interest then */
				if (msg->internal.flags & ERRMEM_FLAG_INITIALIZED) {
					if (msg->internal.seqnum != 1) {
						msg->internal.flags |= ERRMEMD_MESSAGE_MISSING;
						msg->internal.offset = 1;
					}
					d->last_seq = 0;
				} else if (d->flags & ERRMEMD_START) {
					/* After a reboot the sequence number of the currently
					 * incoming message can be checked against the last one
					 * stored in the persistent storage but only in case
					 * the persistent storage was not erased at some point
					 * in time during startup */
					if (d->backends && d->backends && d->backends->get_type) {
						/* we have a block device if condition expression is true */
						if (!((d->backends->get_type(d->backends)) &
							  (FILE_FS | BUFFER))) {
							if (d->backends && d->backends->get_last_seq_nr)
								d->last_seq = d->backends->get_last_seq_nr(d->backends);
						}
					}
				}
				/* When the sequence shall be checked, that means the sequence
				 * number of the last stored message shall be compared with the
				 * one of the currently incoming message, it has to be checked
				 * that both messages belong to the same context of the
				 * current streaming. What does that mean?
				 * When the first message arrives in the daemon the are two
				 * possibilities. There is a preceding message or there is
				 * no preceding message. If the frontend is newly initialized
				 * because of a power down or because of the fact that the
				 * RAM area could not be recovered after a reboot, there is no
				 * preceding message. This incoming message has to have the
				 * sequence number 1. The last message stored in the last
				 * life cycle of the daemon (normally the last stored message
				 * in the persistent storage) is not of interest regarding a 
				 * check against this incoming message because it is not
				 * possible to ensure a relationship between these both
				 * messages. Messages written into the driver and not
				 * transferred to the daemon before the target starts or
				 * restarts are lost when the frontend is newly initialized.
				 * Only in case the driver could recover the RAM area after
				 * a restart, the first incoming message can be compared
				 * with the last message stored in the persistent storage
				 * because the driver then delivers the message which was
				 * stored in the RAM area directly behind the one which could
				 * be delivered to the daemon as the last message before the
				 * target reboots.
				 * All other succeeding messages in this life cycle of the
				 * daemon have a preceeding message, this is the last one.
				 * If no preceeding message has to be taken into account
				 * d->last_seq is 0 */
				if (d->last_seq) {
					/* we have somehow a sequence number alread stored */
					if (d->ifc->nr_drv_slots > 0) {
						uint32_t cur_used = errmem_get_used_slots(d->ifc);
						/* Do checks only in case we have currently no
						 * overwriting. Otherwise it is normal that messages
						 * are going lost due to overwriting. We are looking
						 * for gaps in a sequence of messages where write- and
						 * read-pointer in the frontend are in the same round.
						 * That is write - read < number of slots - 1 */
						if (cur_used < d->ifc->nr_drv_slots - 1) {
							if (msg->internal.seqnum != d->last_seq + 1) {
								if (msg->internal.seqnum < d->last_seq)
									msg->internal.flags |= ERRMEMD_SEQ_NR_MISSMATCH;
								else if (msg->internal.seqnum == d->last_seq)
									msg->internal.flags |= ERRMEMD_SEQ_NR_TWICE;
								else if ((msg->internal.seqnum >= d->last_seq + 2) &&
									 (!(msg->internal.flags & (ERRMEM_FLAG_ALLOC_FAIL | ERRMEM_FLAG_BUSY))))
									msg->internal.flags |= ERRMEMD_MESSAGE_MISSING;
								else if ((msg->internal.seqnum > d->last_seq + 2) &&
										 (msg->internal.flags & ERRMEM_FLAG_ALLOC_FAIL)) {
									msg->internal.flags |= ERRMEMD_MULTIPLE_ALLOC_FAIL;
									msg->internal.flags &= ~ERRMEM_FLAG_ALLOC_FAIL;
								}
								msg->internal.offset = d->last_seq;
							} else
								/* This is ok. A message can lose while trying to
								 * allocate a slot */
								msg->internal.flags &= ~ERRMEM_FLAG_ALLOC_FAIL;
						}
					}
				}
				if (msg->internal.seqnum > d->last_seq)
					d->last_seq = msg->internal.seqnum;

				/* Store the message in the backend
				 * Output devices always returns the internal sequences number.
				 * Persistent storages may return error in case all tries to
				 * store the message failed.
				 */
				msg->internal.flags |= d->flags;
				unsigned s = ~0;
				if (d->p_mode) {
					s = errmem_backend_store(&(d->backends), &(d->removed),
											 ((d->ifc->maxsize - 
											   ERRMEM_MAX_ENTRY_LENGTH) +
											  msg->length),
											 msg);
					/* Check the situation here - backends still available ? 
					 * If yes, stay in persistent output mode and acknowledge.
					 * If no, change mode to non persistent output mode, check
					 * for specified output backends and consider configuration.
					 */
					if (!d->backends)
						d->p_mode = 0;
				}
				
				/* If the daemon is no longer in persistent mode, then if pure 
				 * output devices are specified, write the message to them.
				 * Do not use else condition here because d->pmode may change in
				 * above if-clause 
				 */
				if (!d->p_mode) {
					if (d->outputs) {
						s = errmem_backend_store(&(d->outputs), &(d->removed),
												 ((d->ifc->maxsize - 
												   ERRMEM_MAX_ENTRY_LENGTH) +
												  msg->length),
												 msg);
						/* set acknowledge mode to configured one for pure output */
						ack = d->out_ack;
					} else {
						err = errmem_disconnect_frontend(d);
						ack = 0;
					}
				}
				
				/* A message will be acknowledged if it is successfully stored
				 * persistently.
				 * A message will be acknowledged if it is stored to an output 
				 * device and acknowledging is requested.
				 */
				if (ack && s && ~s)
					(void)_ioctl(d->ifc->efd,
								 IOCTL_ERRMEM_ACKNOWLEDGE,
								 &s);
				/* Check for criteria to stop dumping because of an incoming
				 * real time signal. Do this check here because this is not a
				 * time critical issue rather than a normal user command.
				 * In case the daemon stores messages from the frontend to pure
				 * output devices and it shall stop processing, diconnect
				 * the frontend connection here and wait for new incoming
				 * commands.
				 */
				if (!d->p_mode) { 
					if (break_dump) {
						if (d->outputs) {
							err = errmem_disconnect_frontend(d);
							if (!err)
								err = -ECANCELED;					
						}
						break_dump = 0;
					}
				}
				d->flags &= ~d->flags;				
			} else if (0 == err) {
				/* Normally the driver has to signal that no more messages are
				 * available right now. At this point in time it returns 0 if
				 * no more message is available or EBUSY if write slot
				 * is reached by read pointer
				 */
				/* In case we dump to pure output devices in mode once, we stop now */
				if (!d->p_mode && d->out_once) {
					err = errmem_disconnect_frontend(d);
					if (err < 0)
						syslog(LOG_CRIT, "%s %s", "errmem_flush: "
						       "can't disconnect frontend after dump in "
						       "mode- error = ", strerror(-err));
				}
			} else {
				err = -errno;
				syslog(LOG_INFO, "%s %s", "errmem_flush: "
					   "Result of IOCTL read request: error = ",
					   strerror(-err));
				if (!d->p_mode && d->out_once)
					(void)errmem_disconnect_frontend(d);
			}
		} 
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "errmem_flush: Parameter check failed - error = ", strerror(-err));
	}
	return err;
}

/* The frontend handling covers different scenarios. To understand this
 * it is essential to know that the daemon distinguishes two different
 * kinds of backends, backends representing persistent storages (d->backends) 
 * and backends representing ordinary output devices like TRACE, DLT or
 * consoles (d->outputs) which cannot store messages persistently, at least
 * from the component point of view. For sure these components can write log
 * files but the daemon does not know anything about these logs. The daemon
 * writes to an interface wihtout having the data under contol afterwards.
 * All kinds of backends can be configured with the help of parameters in the 
 * commandline which is passed to the daemon at startup. The daemon sets
 * up the configured backends if possible. Output devices can be additionally
 * configured during runtime using the socket interface. Persistent storages
 * cannot be configured during runtime (at least not now).
 * The handling of messages depends on the backend configuration and the
 * requested daemon status. The daemon can be instructed through the socket
 * interface to cease any access to persistent storages. This status is
 * represented by d->stop (false => access allowed; true => access not allowed).
 * Scenarios:
 * 1.) Only persistent backends are configured, having at least one accessible
       backend available and stop is false:
 *     ==> Messages from frontend will be read, stored and acknowledged as long
 *         as at least one persistent backend is accessible. As soon as the
 *         last available persistent backend is no longer available due to
 *         read-, write- or seek-error the connection to the frontend is closed.
 *         Note: That means the file handle refering to the frontend will
 *               be deleted from epoll instance. It will not be closed.
 * 2.) Only persistent backends are configured, having at least one accessible
 *     backend available and stop is true
 *     ==> Messages from frontend will neither be read/stored
 *         nor acknowledged. Connection to frontend will be closed if not
 *         already done.
 *         Note: That means the file handle refering to the frontend will
 *               be deleted from epoll instance. It will not be closed.
 * 3.) Persistent backends as well as pure output backends are configured and
 *     stop is false
 *     ==> As long as at least one persistent backend is accessible/available
 *         to the daemon messages will be read from the frontend, stored into
 *         all accessible backends and acknowledged to the frontend. The
 *         configured output devices will be completely ignored.
 *         As soon as the last available persistent backend cease to work
 *         (read- / write- / seek-error) the output is automatically redirected
 *         to the configured output devices. Messages will no longer be
 *         acknowledged to the frontend.  
 * 4.) Persistent backends as well as pure output backends are configured and
 *     stop is true
 *     ==> Messages will be read from the frontend, written to each configured
 *         output device and will not be acknowledged to the frontend. All
 *         persistent backends will be ignored. 
 * 5.) Persistent backends may be configured but no persistent backend is still
 *     available/accessible and no output device is configured. It is of no
 *     interest whether stop is false or true.
 *     ==> The connection to the frontend is closed.
 *         Note: That means the file handle refering to the frontend is not
 *               added to the epoll instance. It isn't closed. It refers still
 *               to the error memory frontend.
 *     As soon as a new output device is configured (through socket interface)
 *     the daemon connects to the frontend if not already connected (adds
 *     file handle to epoll instance), reads messages from the frontend and
 *     writes them to the output backends. These messages will not be
 *     acknowledged towards the error memory frontend. The stop status does not
 *     have any impact here.
 */
int32_t errmem_handle_frontend(Errmemd_t *d, ErrmemConnect_t* c)
{
	int32_t err = 0;
	c = c;
	if (d)
		err = errmem_flush(d);
	else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s", "errmem_handle_frontend: Parameter check failed.");
	}
	return err;
}

// errmem_destroy - de-initialize the error memory interface
void errmem_destroy(ErrmemInterface_t* b)
{
	if (b) {
		if (b->efd>0)
			_close(b->efd);

		if (b->name)
			free(b->name);

		if (b->buffer)
			free(b->buffer);

		free(b);
	}
}

// errmem_fill - helper routine to put stuff into /dev/errmem
void errmem_fill(const char* errmem, const char* source)
{
	int fd = _open(errmem, O_WRONLY);
	union {
		struct errmem_message msg;
		unsigned char buffer[4096];
	} m;

	char buffer[sizeof(m.buffer) - (sizeof(m.msg) - sizeof(m.msg.message))];
	int in = source && strcmp(source, "-") ? open(source, O_RDONLY) : 0;
	int rc;

	do {
		rc = read(in, buffer, sizeof(buffer));
		if (rc>0) {
			buffer[rc] = '\0';
			char* t = strtok(buffer, "\n");
			while (rc>0 && t) {
				strcpy((char *) m.msg.message, t);
				m.msg.length = strlen(t);
				m.msg.message[m.msg.length++] = '\n';
				m.msg.message[m.msg.length] = '\0';
				rc = _write(fd, &m.msg, sizeof(m.msg));
				t = strtok(NULL, "\n");
			}
		}
	} while (rc>0);

	if (in>0)
		close(in);
}

// errmem_get_kernel_message_level - get a driver parameter
unsigned errmem_get_kernel_message_level(ErrmemInterface_t* b)
{
	if (b) {
		unsigned long value;
		int rc = _ioctl(b->efd,
				IOCTL_ERRMEM_GET_KERNEL_MSG_LEVEL, &value);
		return rc>=0 ? (unsigned) value : (unsigned) ~0;
	}

	return ~0;
}

// errmem_get_watermark_low - get a driver parameter
unsigned errmem_get_watermark_low(ErrmemInterface_t* b)
{
	if (b) {
		unsigned long value;
		int rc = _ioctl(b->efd,
				IOCTL_ERRMEM_GET_WATERMARK_LOW, &value);
		return rc>=0 ? (unsigned) value : (unsigned) ~0;
	}

	return ~0;
}

// errmem_get_watermark_high - get a driver parameter
unsigned errmem_get_watermark_high(ErrmemInterface_t* b)
{
	if (b) {
		unsigned long value;
		int rc = _ioctl(b->efd,
				IOCTL_ERRMEM_GET_WATERMARK_HIGH, &value);
		return rc>=0 ? (unsigned) value : (unsigned) ~0;
	}

	return ~0;
}

// errmem_get_used_slots - get a driver parameter
unsigned errmem_get_used_slots(ErrmemInterface_t* b)
{
	if (b) {
		unsigned long value;
		int rc = _ioctl(b->efd,
				IOCTL_ERRMEM_GET_USED_SLOTS, &value);
		return rc>=0 ? (unsigned) value : (unsigned) ~0;
	}
	return 0;
}

// errmem_get_slotsize - get a driver parameter
unsigned errmem_get_slotsize(ErrmemInterface_t* b)
{
	if (b) {
		unsigned long value;
		int rc = _ioctl(b->efd,
				IOCTL_ERRMEM_GET_SLOTSIZE, &value);
		return rc>=0 ? (unsigned) value : (unsigned) ~0;
	}
	return 0;
}

// errmem_set_kernel_message_level - set a driver parameter
void errmem_set_kernel_message_level(ErrmemInterface_t* b, unsigned value)
{
	if (b) {
		_ioctl(b->efd,
		       IOCTL_ERRMEM_SET_KERNEL_MSG_LEVEL, &value);
	}
}

// errmem_set_watermark_low - set a driver parameter
int32_t errmem_set_watermark_low(ErrmemInterface_t* b, unsigned value)
{
	int32_t err = 0;
	if (b) {
		err = _ioctl(b->efd,
					 IOCTL_ERRMEM_SET_WATERMARK_LOW, &value);
		if (err < 0) {
			err = -errno;
			syslog(LOG_CRIT, "%s %s", "Failed to set low watermark - error = ",
				   strerror(-err));
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "Errmem Interface handle invalid - error = ",
			   strerror(-err));
	}
	return err;
}

// errmem_set_watermark_high - set a driver parameter
int32_t errmem_set_watermark_high(ErrmemInterface_t* b, unsigned value)
{
	int32_t err = 0;
	if (b) {
		err = _ioctl(b->efd,
					 IOCTL_ERRMEM_SET_WATERMARK_HIGH, &value);
		if (err < 0) {
			err = -errno;
			syslog(LOG_CRIT, "%s %s", "Failed to set high watermark - error = ",
				   strerror(-err));
		}
	} else {
		err = -EINVAL;
		syslog(LOG_CRIT, "%s %s", "Errmem Interface handle invalid - error = ",
			   strerror(-err));
	}
	return err;
}
